<?php
/**
 * DrawTest.php
 *
 * @since       2011-05-23
 * @category    Library
 * @package     PdfGraph
 * @author      Nicola Asuni <info@tecnick.com>
 * @copyright   2011-2015 Nicola Asuni - Tecnick.com LTD
 * @license     http://www.gnu.org/copyleft/lesser.html GNU-LGPL v3 (see LICENSE.TXT)
 * @link        https://github.com/tecnickcom/tc-lib-pdf-graph
 *
 * This file is part of tc-lib-pdf-graph software library.
 */

namespace Test;

use PHPUnit\Framework\TestCase;

/**
 * Draw Test
 *
 * @since       2011-05-23
 * @category    Library
 * @package     PdfGraph
 * @author      Nicola Asuni <info@tecnick.com>
 * @copyright   2011-2015 Nicola Asuni - Tecnick.com LTD
 * @license     http://www.gnu.org/copyleft/lesser.html GNU-LGPL v3 (see LICENSE.TXT)
 * @link        https://github.com/tecnickcom/tc-lib-pdf-graph
 */
class DrawTest extends TestCase
{
    protected $obj = null;
    protected $style = array();

    public function setUp()
    {
        //$this->markTestSkipped(); // skip this test
        $this->obj = new \Com\Tecnick\Pdf\Graph\Draw(
            0.75,
            80,
            100,
            new \Com\Tecnick\Color\Pdf(),
            new \Com\Tecnick\Pdf\Encrypt\Encrypt(),
            false
        );
        $this->style = array(
            'lineWidth'  => 3,
            'lineCap'    => 'round',
            'lineJoin'   => 'bevel',
            'miterLimit' => 11,
            'dashArray'  => array(5, 7),
            'dashPhase'  => 1,
            'lineColor'  => 'greenyellow',
            'fillColor'  => '["RGB",0.250000,0.500000,0.750000]',
        );
    }

    public function testGetLine()
    {
        $res = $this->obj->getLine(3, 5, 7, 11);
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getLine(3, 5, 7, 11, $this->style);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'S'."\n",
            $res
        );
    }

    public function testGetCurve()
    {
        $res = $this->obj->getCurve(3, 5, 7, 11, 13, 17, 19, 23);
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 9.750000 62.250000 14.250000 57.750000 c'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getCurve(3, 5, 7, 11, 13, 17, 19, 23, 'B*', $this->style);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 9.750000 62.250000 14.250000 57.750000 c'."\n"
            .'B*'."\n",
            $res
        );
    }

    public function testGetPolycurve()
    {
        $segments = array(
            array(7, 11, 13, 17, 19, 23),
            array(29, 31, 37, 41, 43, 47),
        );

        $res = $this->obj->getPolycurve(3, 5, $segments);
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 9.750000 62.250000 14.250000 57.750000 c'."\n"
            .'21.750000 51.750000 27.750000 44.250000 32.250000 39.750000 c'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getPolycurve(3, 5, $segments, 'B*', $this->style);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 9.750000 62.250000 14.250000 57.750000 c'."\n"
            .'21.750000 51.750000 27.750000 44.250000 32.250000 39.750000 c'."\n"
            .'B*'."\n",
            $res
        );
    }

    public function testGetEllipse()
    {
        $res = $this->obj->getEllipse(3, 5, 7, 0, 13, 61, 349);
        $this->assertEquals(
            '3.697096 76.296624 m'."\n"
            .'2.476319 76.646676 1.169043 76.542151 0.019397 76.002569 c'."\n"
            .'-1.130248 75.462987 -2.045949 74.524165 -2.556702 73.361424 c'."\n"
            .'-3.067455 72.198684 -3.139357 70.889209 -2.758978 69.677538 c'."\n"
            .'-2.378600 68.465867 -1.571192 67.432432 -0.487535 66.770223 c'."\n"
            .'0.596122 66.108014 1.884095 65.860988 3.135859 66.075278 c'."\n"
            .'4.387624 66.289567 5.520087 66.950946 6.321766 67.935905 c'."\n"
            .'7.123445 68.920865 7.541123 70.164022 7.496802 71.433222 c'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getEllipse(3, 5, 7, 11, 13, 61, 349, 'B*', $this->style, 4);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'4.210523 78.087155 m'."\n"
            .'3.610431 78.606045 2.973969 78.977524 2.330748 79.184310 c'."\n"
            .'1.687527 79.391096 1.047679 79.429932 0.440975 79.299009 c'."\n"
            .'-0.165730 79.168086 -0.729735 78.869468 -1.224798 78.417048 c'."\n"
            .'-1.719862 77.964629 -2.138186 77.365534 -2.460309 76.647637 c'."\n"
            .'-2.782431 75.929740 -3.003276 75.104350 -3.112570 74.209869 c'."\n"
            .'-3.221865 73.315388 -3.217885 72.365906 -3.100818 71.405599 c'."\n"
            .'-2.983751 70.445292 -2.755440 69.489287 -2.426508 68.582062 c'."\n"
            .'-2.097575 67.674838 -1.673202 66.830685 -1.173134 66.088880 c'."\n"
            .'-0.673065 65.347074 -0.105178 64.719300 0.504106 64.234767 c'."\n"
            .'1.113389 63.750233 1.754473 63.416572 2.397528 63.249307 c'."\n"
            .'3.040584 63.082042 3.675482 63.083808 4.272683 63.254523 c'."\n"
            .'4.869885 63.425238 5.419982 63.762213 5.897382 64.249769 c'."\n"
            .'6.374782 64.737326 6.771963 65.367784 7.070448 66.111811 c'."\n"
            .'7.368932 66.855838 7.564018 67.701714 7.646628 68.610083 c'."\n"
            .'7.729237 69.518453 7.698070 70.475006 7.554576 71.435240 c'."\n"
            .'B*'."\n",
            $res
        );
    }

    public function testGetCircle()
    {
        $res = $this->obj->getCircle(3, 5, 7, 61, 349);
        $this->assertEquals(
            '4.795251 75.841753 m'."\n"
            .'3.684506 76.457449 2.387223 76.649676 1.145663 76.382537 c'."\n"
            .'-0.095897 76.115398 -1.199317 75.406626 -1.958540 74.388581 c'."\n"
            .'-2.717762 73.370536 -3.082388 72.110797 -2.984325 70.844615 c'."\n"
            .'-2.886263 69.578433 -2.332021 68.389857 -1.425103 67.500851 c'."\n"
            .'-0.518184 66.611844 0.681209 66.081419 1.949096 66.008630 c'."\n"
            .'3.216982 65.935842 4.469198 66.325521 5.471898 67.104897 c'."\n"
            .'6.474597 67.884274 7.161220 69.001612 7.403543 70.248253 c'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getCircle(3, 5, 7, 61, 349, 'B*', $this->style, 4);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'4.795251 75.841753 m'."\n"
            .'4.244239 76.147184 3.642722 76.350953 3.019526 76.443297 c'."\n"
            .'2.396329 76.535640 1.761569 76.515058 1.145663 76.382537 c'."\n"
            .'0.529758 76.250017 -0.057295 76.007708 -0.587363 75.667224 c'."\n"
            .'-1.117431 75.326740 -1.581909 74.893607 -1.958540 74.388581 c'."\n"
            .'-2.335170 73.883556 -2.617839 73.314836 -2.793000 72.709674 c'."\n"
            .'-2.968161 72.104513 -3.032972 71.472735 -2.984325 70.844615 c'."\n"
            .'-2.935679 70.216495 -2.774366 69.602229 -2.508116 69.031254 c'."\n"
            .'-2.241866 68.460279 -1.875001 67.941864 -1.425103 67.500851 c'."\n"
            .'-0.975204 67.059838 -0.449574 66.703386 0.126598 66.448577 c'."\n"
            .'0.702769 66.193768 1.320130 66.044739 1.949096 66.008630 c'."\n"
            .'2.578061 65.972522 3.208421 66.049920 3.809968 66.237117 c'."\n"
            .'4.411515 66.424313 4.974484 66.718269 5.471898 67.104897 c'."\n"
            .'5.969311 67.491526 6.393094 67.964551 6.722938 68.501304 c'."\n"
            .'7.052782 69.038058 7.283333 69.629827 7.403543 70.248253 c'."\n"
            .'B*'."\n",
            $res
        );
    }

    public function testGetPieSector()
    {
        $res = $this->obj->getPieSector(3, 5, 7, 90, 120);
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'2.250000 76.500000 l'."\n"
            .'1.791198 76.500000 1.334369 76.439857 0.891200 76.321111 c'."\n"
            .'0.448031 76.202364 0.022334 76.026034 -0.375000 75.796633 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'B'."\n",
            $res
        );
    }

    public function testGetBasicPolygon()
    {
        $points = array(3, 5, 7, 11, 13, 17, 19, 23);

        $res = $this->obj->getBasicPolygon($points);
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'9.750000 62.250000 l'."\n"
            .'14.250000 57.750000 l'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getBasicPolygon($points, 'B*', $this->style);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'9.750000 62.250000 l'."\n"
            .'14.250000 57.750000 l'."\n"
            .'B*'."\n",
            $res
        );
    }

    public function testGetPolygon()
    {
        $points = array(3, 5, 7, 11, 13, 17, 19, 23);

        $res = $this->obj->getPolygon(array(3, 5, 7, 11));
        $this->assertEquals('', $res);

        $res = $this->obj->getPolygon($points);
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'S'."\n"
            .'5.250000 66.750000 m'."\n"
            .'9.750000 62.250000 l'."\n"
            .'S'."\n"
            .'9.750000 62.250000 m'."\n"
            .'14.250000 57.750000 l'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getPolygon(
            $points,
            'b*',
            array('all' => $this->style, 0 => array('lineColor' => '#ff0000'))
        );
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'9.750000 62.250000 l'."\n"
            .'14.250000 57.750000 l'."\n"
            .'h f*'."\n"
            .'1.000000 0.000000 0.000000 RG'."\n"
            .'2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'S'."\n"
            .'5.250000 66.750000 m'."\n"
            .'9.750000 62.250000 l'."\n"
            .'S'."\n"
            .'9.750000 62.250000 m'."\n"
            .'14.250000 57.750000 l'."\n"
            .'S'."\n"
            .'1.000000 0.000000 0.000000 RG'."\n"
            .'14.250000 57.750000 m'."\n"
            .'2.250000 71.250000 l'."\n"
            .'S'."\n",
            $res
        );
    }

    public function testGetRegularPolygon()
    {
        $res = $this->obj->getRegularPolygon(3, 5, 7, 2);
        $this->assertEquals('', $res);

        $res = $this->obj->getRegularPolygon(3, 5, 7, 5, 11);
        $this->assertEquals(
            '3.251747 66.096457 m'."\n"
            .'7.460867 70.610186 l'."\n"
            .'S'."\n"
            .'7.460867 70.610186 m'."\n"
            .'4.468746 76.008116 l'."\n"
            .'S'."\n"
            .'4.468746 76.008116 m'."\n"
            .'-1.589607 74.830491 l'."\n"
            .'S'."\n"
            .'-1.589607 74.830491 m'."\n"
            .'-2.341753 68.704749 l'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getRegularPolygon(
            3,
            5,
            7,
            5,
            11,
            'b*',
            array('all' => $this->style, 0 => array('lineColor' => '#ff0000')),
            'f',
            array('fillColor' => '#cccccc')
        );
        $this->assertEquals(
            '0.800000 0.800000 0.800000 rg'."\n"
            .'7.500000 71.250000 m'."\n"
            .'7.500000 72.483995 7.064930 73.679340 6.271733 74.624635 c'."\n"
            .'5.478536 75.569930 4.376901 76.205960 3.161653 76.420241 c'."\n"
            .'1.946405 76.634522 0.693671 76.413631 -0.375000 75.796633 c'."\n"
            .'-1.443671 75.179636 -2.261335 74.205182 -2.683386 73.045606 c'."\n"
            .'-3.105437 71.886030 -3.105437 70.613970 -2.683386 69.454394 c'."\n"
            .'-2.261335 68.294818 -1.443671 67.320364 -0.375000 66.703367 c'."\n"
            .'0.693671 66.086369 1.946405 65.865478 3.161653 66.079759 c'."\n"
            .'4.376901 66.294040 5.478536 66.930070 6.271733 67.875365 c'."\n"
            .'7.064930 68.820660 7.500000 70.016005 7.500000 71.250000 c'."\n"
            .'f'."\n"
            .'2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'3.251747 66.096457 m'."\n"
            .'7.460867 70.610186 l'."\n"
            .'4.468746 76.008116 l'."\n"
            .'-1.589607 74.830491 l'."\n"
            .'-2.341753 68.704749 l'."\n"
            .'h f*'."\n"
            .'1.000000 0.000000 0.000000 RG'."\n"
            .'3.251747 66.096457 m'."\n"
            .'7.460867 70.610186 l'."\n"
            .'S'."\n"
            .'7.460867 70.610186 m'."\n"
            .'4.468746 76.008116 l'."\n"
            .'S'."\n"
            .'4.468746 76.008116 m'."\n"
            .'-1.589607 74.830491 l'."\n"
            .'S'."\n"
            .'-1.589607 74.830491 m'."\n"
            .'-2.341753 68.704749 l'."\n"
            .'S'."\n"
            .'1.000000 0.000000 0.000000 RG'."\n"
            .'-2.341753 68.704749 m'."\n"
            .'3.251747 66.096457 l'."\n"
            .'S'."\n",
            $res
        );
    }

    public function testGetStarPolygon()
    {
        $res = $this->obj->getStarPolygon(3, 5, 7, 1, 0);
        $this->assertEquals('', $res);

        $res = $this->obj->getStarPolygon(3, 5, 7, 11, 13);
        $this->assertEquals(
            '2.250000 66.000000 m'."\n"
            .'7.025568 69.069071 l'."\n"
            .'S'."\n"
            .'7.025568 69.069071 m'."\n"
            .'6.217685 74.688019 l'."\n"
            .'S'."\n"
            .'6.217685 74.688019 m'."\n"
            .'0.770904 76.287338 l'."\n"
            .'S'."\n"
            .'0.770904 76.287338 m'."\n"
            .'-2.946563 71.997153 l'."\n"
            .'S'."\n"
            .'-2.946563 71.997153 m'."\n"
            .'-0.588364 66.833419 l'."\n"
            .'S'."\n"
            .'-0.588364 66.833419 m'."\n"
            .'5.088364 66.833419 l'."\n"
            .'S'."\n"
            .'5.088364 66.833419 m'."\n"
            .'7.446563 71.997153 l'."\n"
            .'S'."\n"
            .'7.446563 71.997153 m'."\n"
            .'3.729096 76.287338 l'."\n"
            .'S'."\n"
            .'3.729096 76.287338 m'."\n"
            .'-1.717685 74.688019 l'."\n"
            .'S'."\n"
            .'-1.717685 74.688019 m'."\n"
            .'-2.525568 69.069071 l'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getStarPolygon(
            3,
            5,
            7,
            11,
            13,
            61,
            'b*',
            array('all' => $this->style, 0 => array('lineColor' => '#ff0000')),
            'f',
            array('fillColor' => '#cccccc')
        );
        $this->assertEquals(
            '0.800000 0.800000 0.800000 rg'."\n"
            .'7.500000 71.250000 m'."\n"
            .'7.500000 72.483995 7.064930 73.679340 6.271733 74.624635 c'."\n"
            .'5.478536 75.569930 4.376901 76.205960 3.161653 76.420241 c'."\n"
            .'1.946405 76.634522 0.693671 76.413631 -0.375000 75.796633 c'."\n"
            .'-1.443671 75.179636 -2.261335 74.205182 -2.683386 73.045606 c'."\n"
            .'-3.105437 71.886030 -3.105437 70.613970 -2.683386 69.454394 c'."\n"
            .'-2.261335 68.294818 -1.443671 67.320364 -0.375000 66.703367 c'."\n"
            .'0.693671 66.086369 1.946405 65.865478 3.161653 66.079759 c'."\n"
            .'4.376901 66.294040 5.478536 66.930070 6.271733 67.875365 c'."\n"
            .'7.064930 68.820660 7.500000 70.016005 7.500000 71.250000 c'."\n"
            .'f'."\n"
            .'2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'6.841753 68.704749 m'."\n"
            .'6.472725 74.369471 l'."\n"
            .'1.166613 76.387000 l'."\n"
            .'-2.872835 72.398504 l'."\n"
            .'-0.922818 67.067211 l'."\n"
            .'4.736763 66.626310 l'."\n"
            .'7.488895 71.591288 l'."\n"
            .'4.115869 76.157243 l'."\n"
            .'-1.438675 74.985796 l'."\n"
            .'-2.680531 69.446569 l'."\n"
            .'1.842242 66.015859 l'."\n"
            .'h f*'."\n"
            .'1.000000 0.000000 0.000000 RG'."\n"
            .'6.841753 68.704749 m'."\n"
            .'6.472725 74.369471 l'."\n"
            .'S'."\n"
            .'6.472725 74.369471 m'."\n"
            .'1.166613 76.387000 l'."\n"
            .'S'."\n"
            .'1.166613 76.387000 m'."\n"
            .'-2.872835 72.398504 l'."\n"
            .'S'."\n"
            .'-2.872835 72.398504 m'."\n"
            .'-0.922818 67.067211 l'."\n"
            .'S'."\n"
            .'-0.922818 67.067211 m'."\n"
            .'4.736763 66.626310 l'."\n"
            .'S'."\n"
            .'4.736763 66.626310 m'."\n"
            .'7.488895 71.591288 l'."\n"
            .'S'."\n"
            .'7.488895 71.591288 m'."\n"
            .'4.115869 76.157243 l'."\n"
            .'S'."\n"
            .'4.115869 76.157243 m'."\n"
            .'-1.438675 74.985796 l'."\n"
            .'S'."\n"
            .'-1.438675 74.985796 m'."\n"
            .'-2.680531 69.446569 l'."\n"
            .'S'."\n"
            .'-2.680531 69.446569 m'."\n"
            .'1.842242 66.015859 l'."\n"
            .'S'."\n"
            .'1.000000 0.000000 0.000000 RG'."\n"
            .'1.842242 66.015859 m'."\n"
            .'6.841753 68.704749 l'."\n"
            .'S'."\n",
            $res
        );
    }

    public function testGetBasicRect()
    {
        $res = $this->obj->getBasicRect(3, 5, 7, 11);
        $this->assertEquals(
            '2.250000 71.250000 5.250000 -8.250000 re'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getBasicRect(3, 5, 7, 11, 'b*', $this->style);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 5.250000 -8.250000 re'."\n"
            .'b*'."\n",
            $res
        );
    }

    public function testGetRect()
    {
        $res = $this->obj->getRect(3, 5, 7, 11);
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'7.500000 71.250000 l'."\n"
            .'S'."\n"
            .'7.500000 71.250000 m'."\n"
            .'7.500000 63.000000 l'."\n"
            .'S'."\n"
            .'7.500000 63.000000 m'."\n"
            .'2.250000 63.000000 l'."\n"
            .'S'."\n"
            .'2.250000 63.000000 m'."\n"
            .'2.250000 71.250000 l'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getRect(
            3,
            5,
            7,
            11,
            'b*',
            array('all' => $this->style, 1 => array('lineColor' => '#ff0000'))
        );
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 m'."\n"
            .'7.500000 71.250000 l'."\n"
            .'7.500000 63.000000 l'."\n"
            .'2.250000 63.000000 l'."\n"
            .'2.250000 71.250000 l'."\n"
            .'h f*'."\n"
            .'2.250000 71.250000 m'."\n"
            .'7.500000 71.250000 l'."\n"
            .'S'."\n"
            .'1.000000 0.000000 0.000000 RG'."\n"
            .'7.500000 71.250000 m'."\n"
            .'7.500000 63.000000 l'."\n"
            .'S'."\n"
            .'7.500000 63.000000 m'."\n"
            .'2.250000 63.000000 l'."\n"
            .'S'."\n"
            .'2.250000 63.000000 m'."\n"
            .'2.250000 71.250000 l'."\n"
            .'S'."\n",
            $res
        );
    }

    public function testGetRoundedRect()
    {
        $res = $this->obj->getRoundedRect(3, 5, 137, 241, 13, 17);
        $this->assertEquals(
            '12.000000 71.250000 m'."\n"
            .'95.250000 71.250000 l'."\n"
            .'100.634776 71.250000 105.000000 65.541631 105.000000 58.500000 c'."\n"
            .'105.000000 -96.750000 l'."\n"
            .'105.000000 -103.791631 100.634776 -109.500000 95.250000 -109.500000 c'."\n"
            .'12.000000 -109.500000 l'."\n"
            .'6.615224 -109.500000 2.250000 -103.791631 2.250000 -96.750000 c'."\n"
            .'2.250000 58.500000 l'."\n"
            .'2.250000 65.541631 6.615224 71.250000 12.000000 71.250000 c'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getRoundedRect(3, 5, 137, 241, 13, 17, '0000');
        $this->assertEquals(
            '2.250000 71.250000 102.750000 -180.750000 re'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getRoundedRect(3, 5, 137, 241, 13, 17, '1010', 'b*', $this->style);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 m'."\n"
            .'95.250000 71.250000 l'."\n"
            .'100.634776 71.250000 105.000000 65.541631 105.000000 58.500000 c'."\n"
            .'105.000000 -96.750000 l'."\n"
            .'105.000000 -109.500000 l'."\n"
            .'12.000000 -109.500000 l'."\n"
            .'6.615224 -109.500000 2.250000 -103.791631 2.250000 -96.750000 c'."\n"
            .'2.250000 58.500000 l'."\n"
            .'2.250000 71.250000 l'."\n"
            .'b*'."\n",
            $res
        );

        $res = $this->obj->getRoundedRect(3, 5, 137, 241, 13, 17, '0101');
        $this->assertEquals(
            '12.000000 71.250000 m'."\n"
            .'95.250000 71.250000 l'."\n"
            .'105.000000 71.250000 l'."\n"
            .'105.000000 -96.750000 l'."\n"
            .'105.000000 -103.791631 100.634776 -109.500000 95.250000 -109.500000 c'."\n"
            .'12.000000 -109.500000 l'."\n"
            .'2.250000 -109.500000 l'."\n"
            .'2.250000 58.500000 l'."\n"
            .'2.250000 65.541631 6.615224 71.250000 12.000000 71.250000 c'."\n"
            .'S'."\n",
            $res
        );
    }

    public function testGetArrow()
    {
        $res = $this->obj->getArrow(3, 5, 7, 11);
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'S'."\n"
            .'4.048317 70.302247 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'2.433189 69.225495 l'."\n"
            .'S'."\n",
            $res
        );

        $res = $this->obj->getArrow(3, 5, 7, 11, 1, 6, 17, $this->style);
        $this->assertEquals(
            '2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'2.250000 71.250000 m'."\n"
            .'4.001925 68.622113 l'."\n"
            .'S'."\n"
            .'2.250000 w'."\n"
            .'1 J'."\n"
            .'2 j'."\n"
            .'8.250000 M'."\n"
            .'[3.750000 5.250000] 0.750000 d'."\n"
            .'0.678431 1.000000 0.184314 RG'."\n"
            .'0.250000 0.500000 0.750000 rg'."\n"
            .'3.957625 71.060425 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'1.768213 69.600818 l'."\n"
            .'s'."\n",
            $res
        );

        $res = $this->obj->getArrow(3, 5, 7, 11, 2, 6, 17, array());
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'3.308549 69.662176 l'."\n"
            .'S'."\n"
            .'3.957625 71.060425 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'1.768213 69.600818 l'."\n"
            .'b'."\n",
            $res
        );

        $res = $this->obj->getArrow(3, 5, 7, 11, 3, 6, 17, array());
        $this->assertEquals(
            '2.250000 71.250000 m'."\n"
            .'3.308549 69.662176 l'."\n"
            .'S'."\n"
            .'3.957625 71.060425 m'."\n"
            .'5.250000 66.750000 l'."\n"
            .'1.768213 69.600818 l'."\n"
            .'f'."\n",
            $res
        );
    }

    public function testGetRegistrationMark()
    {
        $res = $this->obj->getRegistrationMark(3, 5, 7, true, 'All');
        $this->assertEquals(
            'q'."\n"
            .'0.500000 w'."\n"
            .'0 J'."\n"
            .'0 j'."\n"
            .'10.000000 M'."\n"
            .'/CS1 CS 1.000000 SCN'."\n"
            .'/CS1 cs 1.000000 scn'."\n"
            .'1.000000 1.000000 1.000000 1.000000 k'."\n"
            .'2.250000 71.250000 m'."\n"
            .'2.250000 76.500000 l'."\n"
            .'1.328501 76.500000 0.423041 76.257383 -0.375000 75.796633 c'."\n"
            .'-1.173041 75.335884 -1.835884 74.673041 -2.296633 73.875000 c'."\n"
            .'-2.757383 73.076959 -3.000000 72.171499 -3.000000 71.250000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'2.250000 71.250000 m'."\n"
            .'2.250000 66.000000 l'."\n"
            .'3.171499 66.000000 4.076959 66.242617 4.875000 66.703367 c'."\n"
            .'5.673041 67.164116 6.335884 67.826959 6.796633 68.625000 c'."\n"
            .'7.257383 69.423041 7.500000 70.328501 7.500000 71.250000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'7.500000 71.250000 m'."\n"
            .'7.500000 71.583451 7.468232 71.916144 7.405126 72.243569 c'."\n"
            .'7.342020 72.570994 7.247863 72.891663 7.123932 73.201228 c'."\n"
            .'7.000001 73.510793 6.846858 73.807848 6.666581 74.088364 c'."\n"
            .'6.486304 74.368881 6.279711 74.631585 6.049604 74.872915 c'."\n"
            .'5.819496 75.114244 5.566920 75.333103 5.295299 75.526524 c'."\n"
            .'5.023678 75.719944 4.734246 75.887048 4.430929 76.025568 c'."\n"
            .'4.127611 76.164088 3.811786 76.273397 3.487734 76.352011 c'."\n"
            .'3.163683 76.430625 2.832878 76.478187 2.499805 76.494054 c'."\n"
            .'2.166732 76.509920 1.832904 76.494018 1.502847 76.446563 c'."\n"
            .'1.172790 76.399108 0.848004 76.320315 0.532893 76.211254 c'."\n"
            .'0.217782 76.102193 -0.086223 75.963359 -0.375000 75.796633 c'."\n"
            .'-0.663777 75.629908 -0.936014 75.436049 -1.188019 75.217685 c'."\n"
            .'-1.440024 74.999322 -1.670653 74.757445 -1.876779 74.495335 c'."\n"
            .'-2.082904 74.233225 -2.263590 73.952072 -2.416386 73.655689 c'."\n"
            .'-2.569182 73.359306 -2.693394 73.049039 -2.787338 72.729096 c'."\n"
            .'-2.881282 72.409152 -2.944531 72.080985 -2.976228 71.749044 c'."\n"
            .'-3.007924 71.417103 -3.007924 71.082897 -2.976228 70.750956 c'."\n"
            .'-2.944531 70.419015 -2.881282 70.090848 -2.787338 69.770904 c'."\n"
            .'-2.693394 69.450961 -2.569182 69.140694 -2.416386 68.844311 c'."\n"
            .'-2.263590 68.547928 -2.082904 68.266775 -1.876779 68.004665 c'."\n"
            .'-1.670653 67.742555 -1.440024 67.500678 -1.188019 67.282315 c'."\n"
            .'-0.936014 67.063951 -0.663777 66.870092 -0.375000 66.703367 c'."\n"
            .'-0.086223 66.536641 0.217782 66.397807 0.532893 66.288746 c'."\n"
            .'0.848004 66.179685 1.172790 66.100892 1.502847 66.053437 c'."\n"
            .'1.832904 66.005982 2.166732 65.990080 2.499805 66.005946 c'."\n"
            .'2.832878 66.021813 3.163683 66.069375 3.487734 66.147989 c'."\n"
            .'3.811786 66.226603 4.127611 66.335912 4.430929 66.474432 c'."\n"
            .'4.734246 66.612952 5.023678 66.780056 5.295299 66.973476 c'."\n"
            .'5.566920 67.166897 5.819496 67.385756 6.049604 67.627085 c'."\n"
            .'6.279711 67.868415 6.486304 68.131119 6.666581 68.411636 c'."\n"
            .'6.846858 68.692152 7.000001 68.989207 7.123932 69.298772 c'."\n"
            .'7.247863 69.608337 7.342020 69.929006 7.405126 70.256431 c'."\n"
            .'7.468232 70.583856 7.500000 70.916549 7.500000 71.250000 c'."\n"
            .'S'."\n"
            .'0.000000 0.000000 0.000000 0.000000 k'."\n"
            .'2.250000 71.250000 m'."\n"
            .'2.250000 73.875000 l'."\n"
            .'1.789251 73.875000 1.336521 73.753691 0.937500 73.523317 c'."\n"
            .'0.538479 73.292942 0.207058 72.961521 -0.023317 72.562500 c'."\n"
            .'-0.253691 72.163479 -0.375000 71.710749 -0.375000 71.250000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'2.250000 71.250000 m'."\n"
            .'2.250000 68.625000 l'."\n"
            .'2.710749 68.625000 3.163479 68.746309 3.562500 68.976683 c'."\n"
            .'3.961521 69.207058 4.292942 69.538479 4.523317 69.937500 c'."\n"
            .'4.753691 70.336521 4.875000 70.789251 4.875000 71.250000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'4.875000 71.250000 m'."\n"
            .'4.875000 71.416725 4.859116 71.583072 4.827563 71.746785 c'."\n"
            .'4.796010 71.910497 4.748931 72.070831 4.686966 72.225614 c'."\n"
            .'4.625000 72.380396 4.548429 72.528924 4.458291 72.669182 c'."\n"
            .'4.368152 72.809440 4.264856 72.940793 4.149802 73.061457 c'."\n"
            .'4.034748 73.182122 3.908460 73.291552 3.772649 73.388262 c'."\n"
            .'3.636839 73.484972 3.492123 73.568524 3.340464 73.637784 c'."\n"
            .'3.188806 73.707044 3.030893 73.761698 2.868867 73.801005 c'."\n"
            .'2.706842 73.840312 2.541439 73.864094 2.374903 73.872027 c'."\n"
            .'2.208366 73.879960 2.041452 73.872009 1.876424 73.848281 c'."\n"
            .'1.711395 73.824554 1.549002 73.785158 1.391447 73.730627 c'."\n"
            .'1.233891 73.676097 1.081888 73.606679 0.937500 73.523317 c'."\n"
            .'0.793112 73.439954 0.656993 73.343025 0.530991 73.233843 c'."\n"
            .'0.404988 73.124661 0.289673 73.003722 0.186611 72.872667 c'."\n"
            .'0.083548 72.741612 -0.006795 72.601036 -0.083193 72.452845 c'."\n"
            .'-0.159591 72.304653 -0.221697 72.149520 -0.268669 71.989548 c'."\n"
            .'-0.315641 71.829576 -0.347266 71.665493 -0.363114 71.499522 c'."\n"
            .'-0.378962 71.333552 -0.378962 71.166448 -0.363114 71.000478 c'."\n"
            .'-0.347266 70.834507 -0.315641 70.670424 -0.268669 70.510452 c'."\n"
            .'-0.221697 70.350480 -0.159591 70.195347 -0.083193 70.047155 c'."\n"
            .'-0.006795 69.898964 0.083548 69.758388 0.186611 69.627333 c'."\n"
            .'0.289673 69.496278 0.404988 69.375339 0.530991 69.266157 c'."\n"
            .'0.656993 69.156975 0.793112 69.060046 0.937500 68.976683 c'."\n"
            .'1.081888 68.893321 1.233891 68.823903 1.391447 68.769373 c'."\n"
            .'1.549002 68.714842 1.711395 68.675446 1.876424 68.651719 c'."\n"
            .'2.041452 68.627991 2.208366 68.620040 2.374903 68.627973 c'."\n"
            .'2.541439 68.635906 2.706842 68.659688 2.868867 68.698995 c'."\n"
            .'3.030893 68.738302 3.188806 68.792956 3.340464 68.862216 c'."\n"
            .'3.492123 68.931476 3.636839 69.015028 3.772649 69.111738 c'."\n"
            .'3.908460 69.208448 4.034748 69.317878 4.149802 69.438543 c'."\n"
            .'4.264856 69.559207 4.368152 69.690560 4.458291 69.830818 c'."\n"
            .'4.548429 69.971076 4.625000 70.119604 4.686966 70.274386 c'."\n"
            .'4.748931 70.429169 4.796010 70.589503 4.827563 70.753215 c'."\n"
            .'4.859116 70.916928 4.875000 71.083275 4.875000 71.250000 c'."\n"
            .'S'."\n"
            .'0.000000 0.000000 0.000000 0.000000 k'."\n"
            .'2.250000 71.250000 m'."\n"
            .'4.875000 71.250000 l'."\n"
            .'4.875000 71.710749 4.753691 72.163479 4.523317 72.562500 c'."\n"
            .'4.292942 72.961521 3.961521 73.292942 3.562500 73.523317 c'."\n"
            .'3.163479 73.753691 2.710749 73.875000 2.250000 73.875000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'2.250000 71.250000 m'."\n"
            .'-0.375000 71.250000 l'."\n"
            .'-0.375000 70.789251 -0.253691 70.336521 -0.023317 69.937500 c'."\n"
            .'0.207058 69.538479 0.538479 69.207058 0.937500 68.976683 c'."\n"
            .'1.336521 68.746309 1.789251 68.625000 2.250000 68.625000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'4.875000 71.250000 m'."\n"
            .'4.875000 71.416725 4.859116 71.583072 4.827563 71.746785 c'."\n"
            .'4.796010 71.910497 4.748931 72.070831 4.686966 72.225614 c'."\n"
            .'4.625000 72.380396 4.548429 72.528924 4.458291 72.669182 c'."\n"
            .'4.368152 72.809440 4.264856 72.940793 4.149802 73.061457 c'."\n"
            .'4.034748 73.182122 3.908460 73.291552 3.772649 73.388262 c'."\n"
            .'3.636839 73.484972 3.492123 73.568524 3.340464 73.637784 c'."\n"
            .'3.188806 73.707044 3.030893 73.761698 2.868867 73.801005 c'."\n"
            .'2.706842 73.840312 2.541439 73.864094 2.374903 73.872027 c'."\n"
            .'2.208366 73.879960 2.041452 73.872009 1.876424 73.848281 c'."\n"
            .'1.711395 73.824554 1.549002 73.785158 1.391447 73.730627 c'."\n"
            .'1.233891 73.676097 1.081888 73.606679 0.937500 73.523317 c'."\n"
            .'0.793112 73.439954 0.656993 73.343025 0.530991 73.233843 c'."\n"
            .'0.404988 73.124661 0.289673 73.003722 0.186611 72.872667 c'."\n"
            .'0.083548 72.741612 -0.006795 72.601036 -0.083193 72.452845 c'."\n"
            .'-0.159591 72.304653 -0.221697 72.149520 -0.268669 71.989548 c'."\n"
            .'-0.315641 71.829576 -0.347266 71.665493 -0.363114 71.499522 c'."\n"
            .'-0.378962 71.333552 -0.378962 71.166448 -0.363114 71.000478 c'."\n"
            .'-0.347266 70.834507 -0.315641 70.670424 -0.268669 70.510452 c'."\n"
            .'-0.221697 70.350480 -0.159591 70.195347 -0.083193 70.047155 c'."\n"
            .'-0.006795 69.898964 0.083548 69.758388 0.186611 69.627333 c'."\n"
            .'0.289673 69.496278 0.404988 69.375339 0.530991 69.266157 c'."\n"
            .'0.656993 69.156975 0.793112 69.060046 0.937500 68.976683 c'."\n"
            .'1.081888 68.893321 1.233891 68.823903 1.391447 68.769373 c'."\n"
            .'1.549002 68.714842 1.711395 68.675446 1.876424 68.651719 c'."\n"
            .'2.041452 68.627991 2.208366 68.620040 2.374903 68.627973 c'."\n"
            .'2.541439 68.635906 2.706842 68.659688 2.868867 68.698995 c'."\n"
            .'3.030893 68.738302 3.188806 68.792956 3.340464 68.862216 c'."\n"
            .'3.492123 68.931476 3.636839 69.015028 3.772649 69.111738 c'."\n"
            .'3.908460 69.208448 4.034748 69.317878 4.149802 69.438543 c'."\n"
            .'4.264856 69.559207 4.368152 69.690560 4.458291 69.830818 c'."\n"
            .'4.548429 69.971076 4.625000 70.119604 4.686966 70.274386 c'."\n"
            .'4.748931 70.429169 4.796010 70.589503 4.827563 70.753215 c'."\n"
            .'4.859116 70.916928 4.875000 71.083275 4.875000 71.250000 c'."\n"
            .'S'."\n"
            .'Q'."\n",
            $res
        );
    }

    public function testGetCmykRegistrationMark()
    {
        $res = $this->obj->getCmykRegistrationMark(3, 5, 11);
        $this->assertEquals(
            'q'."\n"
            .'1.000000 0.000000 0.000000 0.000000 k'."\n"
            .'2.250000 71.250000 m'."\n"
            .'2.250000 66.300000 l'."\n"
            .'3.118842 66.300000 3.972561 66.528753 4.725000 66.963174 c'."\n"
            .'5.477439 67.397595 6.102405 68.022561 6.536826 68.775000 c'."\n"
            .'6.971247 69.527439 7.200000 70.381158 7.200000 71.250000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'0.000000 1.000000 0.000000 0.000000 k'."\n"
            .'2.250000 71.250000 m'."\n"
            .'7.200000 71.250000 l'."\n"
            .'7.200000 72.118842 6.971247 72.972561 6.536826 73.725000 c'."\n"
            .'6.102405 74.477439 5.477439 75.102405 4.725000 75.536826 c'."\n"
            .'3.972561 75.971247 3.118842 76.200000 2.250000 76.200000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'0.000000 0.000000 1.000000 0.000000 k'."\n"
            .'2.250000 71.250000 m'."\n"
            .'2.250000 76.200000 l'."\n"
            .'1.381158 76.200000 0.527439 75.971247 -0.225000 75.536826 c'."\n"
            .'-0.977439 75.102405 -1.602405 74.477439 -2.036826 73.725000 c'."\n"
            .'-2.471247 72.972561 -2.700000 72.118842 -2.700000 71.250000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'0.000000 0.000000 0.000000 1.000000 k'."\n"
            .'2.250000 71.250000 m'."\n"
            .'-2.700000 71.250000 l'."\n"
            .'-2.700000 70.381158 -2.471247 69.527439 -2.036826 68.775000 c'."\n"
            .'-1.602405 68.022561 -0.977439 67.397595 -0.225000 66.963174 c'."\n"
            .'0.527439 66.528753 1.381158 66.300000 2.250000 66.300000 c'."\n"
            .'2.250000 71.250000 l'."\n"
            .'f'."\n"
            .'0.500000 w'."\n"
            .'0 J'."\n"
            .'0 j'."\n"
            .'10.000000 M'."\n"
            .'/CS5 CS 1.000000 SCN'."\n"
            .'10.500000 71.250000 m'."\n"
            .'10.500000 71.773994 10.450078 72.296798 10.350912 72.811323 c'."\n"
            .'10.251745 73.325847 10.103784 73.829756 9.909035 74.316215 c'."\n"
            .'9.714287 74.802674 9.473634 75.269475 9.190342 75.710287 c'."\n"
            .'8.907049 76.151098 8.582403 76.563920 8.220806 76.943152 c'."\n"
            .'7.859209 77.322384 7.462302 77.666305 7.035470 77.970252 c'."\n"
            .'6.608637 78.274198 6.153815 78.536789 5.677174 78.754464 c'."\n"
            .'5.200532 78.972139 4.704235 79.143909 4.195011 79.267445 c'."\n"
            .'3.685788 79.390982 3.165951 79.465723 2.642551 79.490656 c'."\n"
            .'2.119150 79.515588 1.594563 79.490599 1.075903 79.416027 c'."\n"
            .'0.557242 79.341455 0.046864 79.217638 -0.448311 79.046257 c'."\n"
            .'-0.943485 78.874875 -1.421208 78.656707 -1.875000 78.394710 c'."\n"
            .'-2.328792 78.132713 -2.756593 77.828077 -3.152601 77.484934 c'."\n"
            .'-3.548609 77.141791 -3.911026 76.761699 -4.234938 76.349812 c'."\n"
            .'-4.558850 75.937925 -4.842785 75.496113 -5.082892 75.030369 c'."\n"
            .'-5.323000 74.564624 -5.518191 74.077062 -5.665817 73.574294 c'."\n"
            .'-5.813443 73.071525 -5.912835 72.555834 -5.962643 72.034212 c'."\n"
            .'-6.012452 71.512591 -6.012452 70.987409 -5.962643 70.465788 c'."\n"
            .'-5.912835 69.944166 -5.813443 69.428475 -5.665817 68.925706 c'."\n"
            .'-5.518191 68.422938 -5.323000 67.935376 -5.082892 67.469631 c'."\n"
            .'-4.842785 67.003887 -4.558850 66.562075 -4.234938 66.150188 c'."\n"
            .'-3.911026 65.738301 -3.548609 65.358209 -3.152601 65.015066 c'."\n"
            .'-2.756593 64.671923 -2.328792 64.367287 -1.875000 64.105290 c'."\n"
            .'-1.421208 63.843293 -0.943485 63.625125 -0.448311 63.453743 c'."\n"
            .'0.046864 63.282362 0.557242 63.158545 1.075903 63.083973 c'."\n"
            .'1.594563 63.009401 2.119150 62.984412 2.642551 63.009344 c'."\n"
            .'3.165951 63.034277 3.685788 63.109018 4.195011 63.232555 c'."\n"
            .'4.704235 63.356091 5.200532 63.527861 5.677174 63.745536 c'."\n"
            .'6.153815 63.963211 6.608637 64.225802 7.035470 64.529748 c'."\n"
            .'7.462302 64.833695 7.859209 65.177616 8.220806 65.556848 c'."\n"
            .'8.582403 65.936080 8.907049 66.348902 9.190342 66.789713 c'."\n"
            .'9.473634 67.230525 9.714287 67.697326 9.909035 68.183785 c'."\n"
            .'10.103784 68.670244 10.251745 69.174153 10.350912 69.688677 c'."\n"
            .'10.450078 70.203202 10.500000 70.726006 10.500000 71.250000 c'."\n"
            .'S'."\n"
            .'2.250000 81.975000 m'."\n"
            .'2.250000 76.200000 l'."\n"
            .'S'."\n"
            .'2.250000 66.300000 m'."\n"
            .'2.250000 60.525000 l'."\n"
            .'S'."\n"
            .'-8.475000 71.250000 m'."\n"
            .'-2.700000 71.250000 l'."\n"
            .'S'."\n"
            .'7.200000 71.250000 m'."\n"
            .'12.975000 71.250000 l'."\n"
            .'S'."\n"
            .'Q'."\n",
            $res
        );
    }
}
